const pool = require('../config_db');
const xmlbuilder = require('xmlbuilder');
const axios = require('axios');
const fs = require('fs');
const path = require('path');

async function obterCoordenadas(endereco) {
    const url = `https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(endereco)}`;
    
    try {
        const response = await axios.get(url);
        if (response.data.length > 0) {
            
            return {
                latitude: response.data[0].lat,
                longitude: response.data[0].lon
            };
        }
    } catch (error) {
        console.error("Erro ao obter coordenadas:", error);
    }
    
    return { latitude: null, longitude: null };
}


exports.criarImoveisXML = async (req, res) => {
    const query = `
        SELECT 
            i.*, 
            u.nomeCompleto AS usuario_nome, u.email, u.celular, 
            GROUP_CONCAT(DISTINCT ic.caracteristica_id ORDER BY ic.caracteristica_id SEPARATOR ',') AS caracteristicas,
            GROUP_CONCAT(DISTINCT img.link ORDER BY img.id SEPARATOR ',') AS imagens
        FROM imoveis i
        JOIN usuarios u ON i.id_usuario = u.id
        LEFT JOIN imovel_caracteristicas ic ON i.id = ic.imovel_id
        LEFT JOIN imagens_imovel img ON i.id = img.id_imovel
        WHERE i.disponibilidade LIKE '%Aprovado Zap%'
        GROUP BY i.id, u.id
        ORDER BY i.id DESC;`;

    try {
        const [results] = await pool.query(query);

        for (let imovel of results) {
            const enderecoCompleto = `${imovel.logradouro}, ${imovel.cidade}, ${imovel.estado}, Brasil`;
            const coordenadas = await obterCoordenadas(enderecoCompleto);
            imovel.latitude = coordenadas.latitude;
            imovel.longitude = coordenadas.longitude;
        }

        // Criando a estrutura XML com o nó raiz corretamente configurado
        let xml = xmlbuilder.create('ListingDataFeed', { encoding: 'UTF-8' })
            .att('xmlns', 'http://www.vivareal.com/schemas/1.0/VRSync')
            .att('xmlns:xsi', 'http://www.w3.org/2001/XMLSchema-instance')
            .att('xsi:schemaLocation', 'http://www.vivareal.com/schemas/1.0/VRSync http://xml.vivareal.com/vrsync.xsd');

        // Adicionando Header
        let header = xml.ele('Header');
        header.ele('Provider', 'TSAdministradora de bens');
        header.ele('Email', 'sardinha_adm@yahoo.com.br');
        header.ele('ContactName', 'Thiago Sardinha');
        header.ele('PublishDate', new Date().toISOString());
        header.ele('Telephone', '21 98997-7070');

        // Adicionando Listings
        let listings = xml.ele('Listings');
        results.forEach(imovel => {
            let listing = listings.ele('Listing');
            listing.ele('ListingID', `${imovel.id}`);
            listing.ele('Title', `<![CDATA[${imovel.tituloImovel}]]>`);
            if(imovel.status == 'Venda'){
                listing.ele('TransactionType', 'For Sale');
            }if(imovel.status == 'Aluguel'){
                listing.ele('TransactionType', 'For Rent');
            }if(imovel.status == 'Aluguel e Venda'){
                listing.ele('TransactionType', 'Sale/Rent');
            }
            
            listing.ele('PublicationType', 'STANDARD');

            // Adicionando imagens
            if (imovel.imagens) {
                let media = listing.ele('Media');
                let imagensArray = imovel.imagens.split(',');
                imagensArray.forEach((img, index) => {
                    media.ele('Item', `https://api.apptsadm.com.br/public/${img.trim()}`)
                        .att('medium', 'image')
                        .att('caption', `img${index + 1}`)
                        .att('primary', index === 0 ? 'true' : 'false');
                });
            }

            // Adicionando Details
            let details = listing.ele('Details');
            if(imovel.tipoImovel == 'Casa de Condomínio' || imovel.tipoImovel == 'Casa Residencial' || imovel.tipoImovel == 'Apartamento Residencial' || imovel.tipoImovel == 'Duplex' || imovel.tipoImovel == 'Triplex' || imovel.tipoImovel == 'Cobertura' || imovel.tipoImovel == 'Kitnet' || imovel.tipoImovel == 'Sítio/Fazenda/Chácara' || imovel.tipoImovel == 'Terreno'){
                details.ele('UsageType', 'Residential');
            }else{
                details.ele('UsageType', 'Commercial');
            }

            if(imovel.tipoImovel == 'Casa de Condomínio'){                
                details.ele('PropertyType', 'Residential / Condo');
            }if(imovel.tipoImovel == 'Casa Residencial'){                
                details.ele('PropertyType', 'Residential / Home');
            }if(imovel.tipoImovel == 'Apartamento Residencial'){                
                details.ele('PropertyType', 'Residential / Apartment');
            }if(imovel.tipoImovel == 'Duplex'){                
                details.ele('PropertyType', 'Residential / Sobrado');
            }if(imovel.tipoImovel == 'Triplex'){                
                details.ele('PropertyType', 'Residential / Sobrado');
            }if(imovel.tipoImovel == 'Cobertura'){                
                details.ele('PropertyType', 'Residential / Penthouse');
            }if(imovel.tipoImovel == 'Casa comercial'){                
                details.ele('PropertyType', 'Commercial / Business');
            }if(imovel.tipoImovel == 'Galpão Comercial'){                
                details.ele('PropertyType', 'Commercial / Industrial');
            }if(imovel.tipoImovel == 'Loja Comercial'){                
                details.ele('PropertyType', 'Commercial / Business');
            }if(imovel.tipoImovel == 'Sala Comercial'){                
                details.ele('PropertyType', 'Commercial / Office');
            }if(imovel.tipoImovel == 'Kitnet'){                
                details.ele('PropertyType', 'Residential / Kitnet');
            }if(imovel.tipoImovel == 'Sítio/Fazenda/Chácara'){                
                details.ele('PropertyType', 'Residential / Agricultural');
            }if(imovel.tipoImovel == 'Terreno'){                
                details.ele('PropertyType', 'Residential / Land Lot');
            }

            details.ele('Description', `<![CDATA[${imovel.descricaoImovel}]]>`);

            var resultadoAluguel = "";
            var resultadoVenda = "";

            if(imovel.valorAluguel != null && imovel.valorAluguel != "null"){
                resultadoAluguel = imovel.valorAluguel.split(",")[0];
            }
            if(imovel.valorVenda != null && imovel.valorVenda != "null"){
                resultadoVenda = imovel.valorVenda.split(",")[0];
            }


            if(imovel.status == 'Venda'){
                details.ele('ListPrice', resultadoVenda).att('currency', 'BRL');
            }if(imovel.status == 'Aluguel'){
                details.ele('RentalPrice', resultadoAluguel).att('currency', 'BRL');
            }if(imovel.status == 'Aluguel e Venda'){
                details.ele('ListPrice', resultadoVenda).att('currency', 'BRL');
                details.ele('RentalPrice', resultadoAluguel).att('currency', 'BRL');
            }            

            details.ele('LotArea', imovel.area).att('unit', 'square metres');
            details.ele('LivingArea', imovel.area).att('unit', 'square metres');

            if(imovel.valorCondominio != null && imovel.valorCondominio != "null"){                
                details.ele('PropertyAdministrationFee', '980').att('currency', 'BRL');
            }

            var resultadoValorIptu = "";

            if(imovel.valorIptu != null && imovel.valorIptu != "null"){
                resultadoValorIptu = imovel.valorIptu.split(",")[0];
                details.ele('YearlyTax', resultadoValorIptu).att('currency', 'BRL').att('period', 'Monthly');
            }

            
            details.ele('Bedrooms', imovel.quartos);
            details.ele('Bathrooms', imovel.banheiros);
            details.ele('Suites', imovel.suites);
            details.ele('Garage', imovel.vagas).att('type', 'Parking Space');

            // Adicionando Features
            let features = details.ele('Features');
            if (imovel.caracteristicas) {
                let caracteristicasArray = imovel.caracteristicas.split(',');
                caracteristicasArray.forEach(car => {
                    features.ele('Feature', car.trim());
                });
            }

            // Adicionando Location
            let location = listing.ele('Location').att('displayAddress', 'Street');

            location.ele('Country', 'Brasil').att('abbreviation', 'BR');

            const estados = {
                "AC": "Acre",
                "AL": "Alagoas",
                "AP": "Amapá",
                "AM": "Amazonas",
                "BA": "Bahia",
                "CE": "Ceará",
                "DF": "Distrito Federal",
                "ES": "Espírito Santo",
                "GO": "Goiás",
                "MA": "Maranhão",
                "MT": "Mato Grosso",
                "MS": "Mato Grosso do Sul",
                "MG": "Minas Gerais",
                "PA": "Pará",
                "PB": "Paraíba",
                "PR": "Paraná",
                "PE": "Pernambuco",
                "PI": "Piauí",
                "RJ": "Rio de Janeiro",
                "RN": "Rio Grande do Norte",
                "RS": "Rio Grande do Sul",
                "RO": "Rondônia",
                "RR": "Roraima",
                "SC": "Santa Catarina",
                "SP": "São Paulo",
                "SE": "Sergipe",
                "TO": "Tocantins"
            };
            
            var estadoCompleto = estados[imovel.estado] || imovel.estado;

            location.ele('State', estadoCompleto).att('abbreviation', imovel.estado);
            location.ele('City', imovel.cidade);
            location.ele('Neighborhood', imovel.bairro);
            location.ele('Address', imovel.logradouro);
            location.ele('StreetNumber', imovel.numero);
            if(imovel.complemento != null && imovel.complemento != "null"){
                location.ele('Complement', imovel.complemento);
            }
            
            location.ele('PostalCode', imovel.cep);

            location.ele('Latitude', imovel.latitude || '0.0');
            location.ele('Longitude', imovel.longitude || '0.0');

            
            let contactInfo = listing.ele('ContactInfo');
            contactInfo.ele('Name', `Ts Administradora de Bens`);
            contactInfo.ele('Email', `sardinha_adm@yahoo.com.br`);
            contactInfo.ele('Website', `https://tsadm.com.br`);
            contactInfo.ele('Telephone', `21 98997-7070`);
        });

        // Convertendo o XML para string
        const xmlString = xml.end({ pretty: true });

        const filePath = path.join(__dirname, '../public/xml/imoveis.xml');

        // Salva o arquivo XML
        fs.writeFileSync(filePath, xmlString, 'utf8');
        console.log(`Arquivo XML gerado em: ${filePath}`);

        // Definindo o cabeçalho e enviando o XML como resposta
        res.set('Content-Type', 'application/xml');
        res.send(xmlString);

    } catch (err) {
        console.error("Erro ao gerar XML:", err);
        res.status(500).send("Erro ao gerar o XML.");
    }
};


